#!/bin/sh

# For postmaster startup options, edit $PGDATA/postgresql.conf
#
# Note that PGDATA is set in ~pgsql/.profile, don't try to manipulate it here!
#

PREFIX=/usr/syno/pgsql
PGBIN=${PREFIX}/bin
POSTGRESQL_CONF=${PREFIX}/etc/postgresql.conf
PGHBA_CONF=${PREFIX}/etc/pg_hba.conf
PGSQL_STARTUP_LOG=/var/log/pgsql.startup.log
MY_VERSION="8.3"
MY_PLATFORM=`/bin/get_key_value /etc.defaults/synoinfo.conf unique | cut -d'_' -f2`
CMD_GET_PG_PATH="/usr/syno/bin/servicetool --get-service-path pgsql"
eval $(grep "^runpgsql=" /etc/synoinfo.conf)

IsPassCompatibilityCheck()
{
    PGDATA_PATH=$1

    /usr/syno/pgsql/bin/pg_controldata --check-compatibility ${PGDATA_PATH} 2>/dev/null

    if [ $? -eq 0 ]; then
	return 1 #pass
    fi

    return 0 #no pass
}

FuncInitialEnv()
{
    chmod 755 /var
}

FuncInitialDB()
{
	if [ "${runpgsql}" != "yes" ]; then
		exit 1;
	fi

        FuncInitialEnv

	PGDATA=`$CMD_GET_PG_PATH`
	ret=$?
	if [ $ret -eq 0 ]; then
	    echo "No volume to start pgSQL..."
	    exit 1;
	fi

	if [ -d ${PGDATA} ]; then
	    IsPassCompatibilityCheck ${PGDATA}
	    if [ $? -eq 0 ]; then
		echo "Compatibility check is failed. Removing old databases"
		rm -rf ${PGDATA}/*
	    fi
	fi

	if [ ! -d ${PGDATA}/base ]; then
		echo "Initial PostgreSQL database"
		chown admin:users ${PGDATA}
		su -l admin -c "/usr/syno/pgsql/bin/initdb --locale=POSIX --pgdata=${PGDATA} --auth=md5" >/dev/null 2>&1
		echo "$MY_PLATFORM" > ${SZF_PLATFORM}

		# we have our config files in /usr/syno/pgsql/etc/
		rm ${PGDATA}/postgresql.conf ${PGDATA}/pg_hba.conf

		# sleep 1 to make sure database created and the daemon is down.
		sync
		sleep 1
	fi
}

FuncCheckDB()
{
	PG_UPGRADE_DIR="/.@pgsql_upgrade"
	if [ ! -d "$PG_UPGRADE_DIR" ]; then
		PG_UPGRADE_DIR=`ls -d /volume[0-9]*/.@pgsql_upgrade 2>/dev/null`
	fi

	if [ -d "$PG_UPGRADE_DIR" ]; then
		echo "**********************************************************"
		echo "Found PostgreSQL upgrade dir at $PG_UPGRADE_DIR"
		echo "**********************************************************"
		if [ -s $PG_UPGRADE_DIR/roundcubemail.sql ]; then
			su -l admin -c "/usr/syno/pgsql/bin/createuser -SDR roundcube" >/dev/null 2>&1
		fi
		for sql in `ls $PG_UPGRADE_DIR`;
		do
			echo "=========================================================="
			echo "Restoring $sql ..."
			echo "=========================================================="
			su -l admin -c "/usr/syno/pgsql/bin/psql postgres < $PG_UPGRADE_DIR/$sql" >/dev/null 2>&1
		done
		rm -rf $PG_UPGRADE_DIR
	fi

	#media server
	/usr/syno/bin/mediaserver.sh start
}

case $1 in
start)
	FuncInitialDB
	[ -x ${PGBIN}/pg_ctl ] && {
	echo "Starting PostgreSQL..."

	[ -d ${PGDATA} ] && chown -R admin:users ${PGDATA}
	su -l admin -c \
		"[ -d ${PGDATA} ] && exec ${PREFIX}/bin/pg_ctl -D ${PGDATA} start -s -w -o \"--config_file=${POSTGRESQL_CONF} --hba_file=${PGHBA_CONF}\"" >/dev/null 2>&1
	}
	if [ $? -eq 0 ]; then
		if [ -e ${PGSQL_STARTUP_LOG} ]; then
			rm ${PGSQL_STARTUP_LOG}
		fi
	else
		if [ -e /var/tmp/postmaster.pid ]; then
			$0 stop
		fi
		[ -x ${PGBIN}/pg_ctl ] && {
		su -l admin -c \
			"[ -d ${PGDATA} ] && exec ${PREFIX}/bin/pg_ctl -D ${PGDATA} start -s -w -o \"--config_file=${POSTGRESQL_CONF} --hba_file=${PGHBA_CONF}\"" >${PGSQL_STARTUP_LOG} 2>&1
		}
		if [ $? -eq 0 ]; then
			echo "Starting PostgreSQL with log enabled"
		else
			echo "Failed to start PostgreSQL. Please check log ${PGSQL_STARTUP_LOG}"
			if grep "No space left on device" ${PGSQL_STARTUP_LOG}; then
				synodsmnotify @administrators dsmnotify:system_event widget:pgsql_startup_failed
				logger -p user.err -t `basename $0` "no space left on device result pgsql startup failed"
			fi
		fi
		exit 1;
	fi
	FuncCheckDB
	;;

stop)
	[ -x ${PGBIN}/pg_ctl ] && {
	echo "Stopping PostgreSQL..."
	su -l admin -c "exec ${PREFIX}/bin/pg_ctl stop -s -m fast" >/dev/null 2>&1
	}
	;;

restart)
	FuncInitialDB
	[ -x ${PGBIN}/pg_ctl ] && {
		exec su -l admin -c "exec ${PREFIX}/bin/pg_ctl -D ${PGDATA} restart -s -m fast -o \"--config_file=${POSTGRESQL_CONF} --hba_file=${PGHBA_CONF}\"" >/dev/null 2>&1
	}
	;;

reload)
	FuncInitialDB
	[ -x ${PGBIN}/pg_ctl ] && {
		exec su -l admin -c "exec ${PREFIX}/bin/pg_ctl reload" >/dev/null 2>&1
	}
	;;

status)
	[ -x ${PGBIN}/pg_ctl ] && {
		exec su -l admin -c "exec ${PREFIX}/bin/pg_ctl status" >/dev/null 2>&1
	}
	;;

*)
	echo "usage: `basename $0` {start|stop|restart|reload|status}" >&2
	exit 64
	;;
esac
